const $ = id => document.getElementById(id);

// ══════════════════════════════════════════════════
// ── Tab switching ──
// ══════════════════════════════════════════════════

let translationsLoaded = false;

document.querySelectorAll('.tab-btn').forEach(btn => {
  btn.addEventListener('click', () => {
    document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
    document.querySelectorAll('.tab-pane').forEach(p => p.classList.remove('active'));
    btn.classList.add('active');
    $('tab-' + btn.dataset.tab).classList.add('active');

    if (btn.dataset.tab === 'translations' && !translationsLoaded) {
      loadTranslations();
    }
    if (btn.dataset.tab === 'current') {
      loadQueue();
    }
  });
});

// ══════════════════════════════════════════════════
// ── Provider / Model / Key (Tab 3: Settings) ──
// ══════════════════════════════════════════════════

chrome.storage.sync.get(['provider', 'model'], (data) => {
  const prov = data.provider || DEFAULT_PROVIDER;
  const mod = data.model || DEFAULT_MODEL;
  const radio = document.querySelector(`input[value="${prov}"]`);
  if (radio) radio.checked = true;
  onProviderChange(prov, mod);
});

chrome.storage.local.get(['apiKey'], (data) => {
  if (data.apiKey) $('apiKeyInput').value = data.apiKey;
});

document.querySelectorAll('input[name="provider"]').forEach(r => {
  r.addEventListener('change', (e) => {
    const firstModel = PROVIDERS[e.target.value].models[0].code;
    chrome.storage.sync.set({ provider: e.target.value, model: firstModel });
    onProviderChange(e.target.value, firstModel);
  });
});

function onProviderChange(provider, selectedModel) {
  $('apiKeySection').style.display = PROVIDERS[provider].needsKey ? 'block' : 'none';

  $('serverStatus').style.display = PROVIDERS[provider].needsServer ? 'flex' : 'none';
  if (PROVIDERS[provider].needsServer) checkServer();

  const sel = $('modelSelect');
  sel.innerHTML = '';
  for (const m of PROVIDERS[provider].models) {
    const opt = document.createElement('option');
    opt.value = m.code;
    opt.textContent = m.label;
    if (m.code === selectedModel) opt.selected = true;
    sel.appendChild(opt);
  }
}

let keyTimer;
$('apiKeyInput').addEventListener('input', (e) => {
  clearTimeout(keyTimer);
  keyTimer = setTimeout(() => {
    const val = e.target.value.trim();
    if (val) {
      chrome.storage.local.set({ apiKey: val });
    } else {
      chrome.storage.local.remove('apiKey');
    }
  }, 500);
});

$('toggleKey').addEventListener('click', () => {
  const inp = $('apiKeyInput');
  inp.type = inp.type === 'password' ? 'text' : 'password';
});

$('modelSelect').addEventListener('change', (e) => {
  chrome.storage.sync.set({ model: e.target.value });
});

function checkServer() {
  const row = $('serverStatus');
  const dot = row.querySelector('.dot');
  const label = row.querySelector('span');
  chrome.runtime.sendMessage({ type: 'ping_server' }, (resp) => {
    if (chrome.runtime.lastError || !resp?.ok) {
      row.className = 'status-row status-err';
      dot.className = 'dot dot-red';
      label.textContent = 'Сервер очередей недоступен';
    } else {
      row.className = 'status-row status-ok';
      dot.className = 'dot dot-green';
      label.textContent = 'Сервер очередей работает';
    }
  });
}

// ══════════════════════════════════════════════════
// ── Toggle (Tab 1) ──
// ══════════════════════════════════════════════════

chrome.storage.sync.get(['isEnabled'], (data) => {
  $('toggleEnabled').checked = data.isEnabled !== false;
});
$('toggleEnabled').addEventListener('change', () => {
  chrome.storage.sync.set({ isEnabled: $('toggleEnabled').checked });
});

// ══════════════════════════════════════════════════
// ── Subtitle style (Tab 1) ──
// ══════════════════════════════════════════════════

let colorTouched = false;

function saveSubtitleStyle() {
  const style = {};
  const fontSize = parseInt($('styleFontSize').value);
  if (fontSize >= 14) style.fontSize = fontSize;
  else style.fontSize = 'auto';

  if (colorTouched) style.color = $('styleColor').value;

  style.bgOpacity = parseInt($('styleBgOpacity').value);

  const position = $('stylePosition').value;
  if (position) style.position = position;

  chrome.storage.sync.set({ subtitleStyle: style });
}

chrome.storage.sync.get('subtitleStyle', (data) => {
  const s = data.subtitleStyle || {};
  if (s.fontSize && s.fontSize !== 'auto') {
    $('styleFontSize').value = s.fontSize;
    $('styleFontSizeVal').textContent = s.fontSize + 'px';
  }
  if (s.color) {
    $('styleColor').value = s.color;
    colorTouched = true;
  }
  if (s.bgOpacity !== undefined) {
    $('styleBgOpacity').value = s.bgOpacity;
    $('styleBgOpacityVal').textContent = s.bgOpacity + '%';
  }
  if (s.position) $('stylePosition').value = s.position;
});

$('styleFontSize').addEventListener('input', (e) => {
  const v = parseInt(e.target.value);
  $('styleFontSizeVal').textContent = v < 14 ? 'авто' : v + 'px';
  saveSubtitleStyle();
});

$('styleColor').addEventListener('input', () => {
  colorTouched = true;
  saveSubtitleStyle();
});

$('styleColorReset').addEventListener('click', () => {
  $('styleColor').value = '#ffffff';
  colorTouched = false;
  saveSubtitleStyle();
});

$('styleBgOpacity').addEventListener('input', (e) => {
  $('styleBgOpacityVal').textContent = e.target.value + '%';
  saveSubtitleStyle();
});

$('stylePosition').addEventListener('change', () => saveSubtitleStyle());

// ══════════════════════════════════════════════════
// ── Shared cache settings (Tab 3) ──
// ══════════════════════════════════════════════════

chrome.storage.sync.get(['sharedCacheEnabled', 'sharedCacheUrl', 'sharedCacheApiKey'], (data) => {
  $('toggleSharedCache').checked = data.sharedCacheEnabled !== false;
  if (data.sharedCacheUrl) $('sharedCacheUrl').value = data.sharedCacheUrl;
  if (data.sharedCacheApiKey) $('sharedCacheApiKey').value = data.sharedCacheApiKey;
});

$('toggleSharedCache').addEventListener('change', () => {
  chrome.storage.sync.set({ sharedCacheEnabled: $('toggleSharedCache').checked });
});

let cacheUrlTimer;
$('sharedCacheUrl').addEventListener('input', (e) => {
  clearTimeout(cacheUrlTimer);
  cacheUrlTimer = setTimeout(() => {
    chrome.storage.sync.set({ sharedCacheUrl: e.target.value.trim() });
  }, 500);
});

let cacheKeyTimer;
$('sharedCacheApiKey').addEventListener('input', (e) => {
  clearTimeout(cacheKeyTimer);
  cacheKeyTimer = setTimeout(() => {
    chrome.storage.sync.set({ sharedCacheApiKey: e.target.value.trim() });
  }, 500);
});

// ══════════════════════════════════════════════════
// ── Queue (Tab 1) ──
// ══════════════════════════════════════════════════

function loadQueue() {
  chrome.runtime.sendMessage({ type: 'get_queue_list', limit: 10 }, (resp) => {
    if (chrome.runtime.lastError || resp?.error || !resp?.jobs) {
      $('queueSection').style.display = 'none';
      return;
    }

    const active = resp.jobs.filter(j => j.status === 'pending' || j.status === 'running');
    if (active.length === 0) {
      $('queueSection').style.display = 'none';
      return;
    }

    $('queueSection').style.display = 'block';
    const list = $('queueList');
    list.innerHTML = '';

    for (const job of active) {
      const item = document.createElement('div');
      item.className = 'queue-item';

      const status = document.createElement('span');
      status.className = 'queue-status ' + job.status;
      status.textContent = job.status === 'pending' ? 'ждёт' : 'идёт';

      const title = document.createElement('span');
      title.className = 'queue-title';
      title.textContent = job.title || job.id;

      item.appendChild(status);
      item.appendChild(title);

      if (job.status === 'running' && job.progress_total > 0) {
        const progress = document.createElement('span');
        progress.className = 'queue-progress';
        const pct = Math.round((job.progress_done / job.progress_total) * 100);
        progress.textContent = pct + '%';
        item.appendChild(progress);
      }

      list.appendChild(item);
    }
  });
}

loadQueue();

// ══════════════════════════════════════════════════
// ── Translations list with grouping (Tab 2) ──
// ══════════════════════════════════════════════════

const TRANSLATIONS_LIMIT = 50;
let translationsOffset = 0;
let translationsTotal = 0;
let allTranslations = []; // accumulate for grouping

function loadTranslations(append) {
  if (!append) {
    translationsOffset = 0;
    allTranslations = [];
    $('translationsList').innerHTML = '';
  }

  $('translationsLoading').style.display = 'block';
  $('translationsEmpty').style.display = 'none';
  $('translationsError').style.display = 'none';
  $('loadMoreBtn').style.display = 'none';

  chrome.runtime.sendMessage({
    type: 'get_translations_list',
    limit: TRANSLATIONS_LIMIT,
    offset: translationsOffset,
  }, (resp) => {
    $('translationsLoading').style.display = 'none';
    translationsLoaded = true;

    if (chrome.runtime.lastError) {
      showTranslationsError('Ошибка связи с расширением');
      return;
    }
    if (resp?.error) {
      showTranslationsError(resp.error);
      return;
    }
    if (!resp?.translations) {
      showTranslationsError('Нет данных');
      return;
    }

    translationsTotal = resp.total || 0;

    if (resp.translations.length === 0 && translationsOffset === 0) {
      $('translationsEmpty').style.display = 'block';
      $('translationsCounter').style.display = 'none';
      return;
    }

    allTranslations.push(...resp.translations);
    translationsOffset += resp.translations.length;

    renderGroupedTranslations();

    $('translationsCounter').style.display = 'block';
    $('translationsCounter').textContent = `${allTranslations.length} из ${translationsTotal}`;
    $('loadMoreBtn').style.display = allTranslations.length < translationsTotal ? 'block' : 'none';
  });
}

function renderGroupedTranslations() {
  const list = $('translationsList');
  list.innerHTML = '';

  // Group by normalized title
  const groups = new Map();
  for (const t of allTranslations) {
    const key = t.title || '(без названия)';
    if (!groups.has(key)) groups.set(key, []);
    groups.get(key).push(t);
  }

  for (const [title, entries] of groups) {
    const group = document.createElement('div');
    group.className = 'translation-group';

    const header = document.createElement('div');
    header.className = 'translation-group-header';

    const titleEl = document.createElement('span');
    titleEl.className = 'translation-group-title';
    titleEl.textContent = title;
    header.appendChild(titleEl);

    // Best model among entries (highest rank)
    const bestEntry = entries.reduce((a, b) => (b.model_rank || 0) > (a.model_rank || 0) ? b : a);
    const latestDate = Math.max(...entries.map(e => e.updated_at || 0));
    const bestUrl = entries.find(e => e.page_url)?.page_url || '';

    // Info block: episodes + model badge + date
    const info = document.createElement('span');
    info.className = 'translation-group-info';

    // Episode range or count
    const episodes = entries.map(e => extractEpisode(e.page_url)).filter(Boolean);
    episodes.sort();
    if (entries.length > 1) {
      const epEl = document.createElement('span');
      epEl.className = 'translation-group-episodes';
      epEl.textContent = episodes.length > 0 ? formatEpisodeRange(episodes) : entries.length + ' шт.';
      info.appendChild(epEl);
    } else if (episodes.length === 1) {
      const epEl = document.createElement('span');
      epEl.className = 'translation-group-episodes';
      epEl.textContent = episodes[0];
      info.appendChild(epEl);
    }

    const modelEl = document.createElement('span');
    modelEl.className = 'translation-group-model';
    modelEl.textContent = formatModelName(bestEntry.model);
    info.appendChild(modelEl);

    const dateEl = document.createElement('span');
    dateEl.className = 'translation-group-date';
    dateEl.textContent = formatDate(latestDate);
    info.appendChild(dateEl);

    header.appendChild(info);

    // Click header → open show page
    const showUrl = entries.length > 1 && bestUrl ? stripEpisodeFromUrl(bestUrl) : bestUrl;
    if (showUrl) {
      header.addEventListener('click', () => chrome.tabs.create({ url: showUrl }));
    }

    group.appendChild(header);

    // Sub-list of episodes (only for multi-entry groups)
    if (entries.length > 1) {
      // Sort entries by episode
      entries.sort((a, b) => {
        const ea = extractEpisode(a.page_url);
        const eb = extractEpisode(b.page_url);
        return ea.localeCompare(eb, undefined, { numeric: true });
      });

      const subList = document.createElement('div');
      subList.className = 'translation-group-entries';

      for (const entry of entries) {
        const row = document.createElement('div');
        row.className = 'translation-entry';

        const ep = document.createElement('span');
        ep.className = 'translation-entry-ep';
        const epLabel = extractEpisode(entry.page_url);
        ep.textContent = epLabel || '#' + (entries.indexOf(entry) + 1);

        const model = document.createElement('span');
        model.className = 'translation-entry-model';
        model.textContent = formatModelName(entry.model);

        const date = document.createElement('span');
        date.className = 'translation-entry-date';
        date.textContent = formatDate(entry.updated_at);

        row.appendChild(ep);
        row.appendChild(model);
        row.appendChild(date);

        if (entry.page_url) {
          row.addEventListener('click', () => chrome.tabs.create({ url: entry.page_url }));
        }

        subList.appendChild(row);
      }

      group.appendChild(subList);
    }

    list.appendChild(group);
  }
}

function showTranslationsError(msg) {
  $('translationsError').style.display = 'block';
  $('translationsError').textContent = msg;
}

$('loadMoreBtn').addEventListener('click', () => loadTranslations(true));

// ── Helpers ──

function formatModelName(model) {
  if (!model) return '';
  const name = model.includes('/') ? model.split('/').pop() : model;
  return name
    .replace(/^claude-/, 'Claude ')
    .replace(/^gpt-/, 'GPT-')
    .replace(/^gemini-/, 'Gemini ')
    .replace(/^llama-/, 'Llama ')
    .replace(/-/g, ' ')
    .replace(/\b\w/g, c => c.toUpperCase())
    .replace(/(\d)\.(\d)/g, '$1.$2');
}

function formatDate(ts) {
  if (!ts) return '';
  const now = Date.now() / 1000;
  const diff = now - ts;
  if (diff < 86400) return 'сегодня';
  if (diff < 172800) return 'вчера';
  if (diff < 604800) return Math.floor(diff / 86400) + 'д назад';
  const d = new Date(ts * 1000);
  const months = ['янв', 'фев', 'мар', 'апр', 'май', 'июн', 'июл', 'авг', 'сен', 'окт', 'ноя', 'дек'];
  return d.getDate() + ' ' + months[d.getMonth()];
}

// "https://kino.pub/item/view/120646/s1e2" → "S1E2"
function extractEpisode(url) {
  if (!url) return '';
  const m = url.match(/\/s(\d+)e(\d+)/i);
  if (m) return 'S' + m[1] + 'E' + m[2];
  return '';
}

// ["S1E1", "S1E2", "S1E3", "S1E4"] → "S1 · E1-E4"
// ["S1E1", "S1E3", "S2E1"] → "S1 · E1, E3 | S2 · E1"
function formatEpisodeRange(episodes) {
  if (episodes.length === 0) return '';

  // Parse into {season, episode} and group by season
  const parsed = episodes.map(ep => {
    const m = ep.match(/S(\d+)E(\d+)/i);
    return m ? { s: parseInt(m[1]), e: parseInt(m[2]), raw: ep } : null;
  }).filter(Boolean);

  if (parsed.length === 0) return episodes.join(', ');

  const byS = new Map();
  for (const p of parsed) {
    if (!byS.has(p.s)) byS.set(p.s, []);
    byS.get(p.s).push(p.e);
  }

  const parts = [];
  for (const [s, eps] of byS) {
    eps.sort((a, b) => a - b);
    // Check if consecutive
    const isConsecutive = eps.every((e, i) => i === 0 || e === eps[i - 1] + 1);
    let epStr;
    if (isConsecutive && eps.length > 1) {
      epStr = 'E' + eps[0] + '-E' + eps[eps.length - 1];
    } else {
      epStr = eps.map(e => 'E' + e).join(', ');
    }
    parts.push('S' + s + ' \u00b7 ' + epStr);
  }
  return parts.join(' | ');
}

// "https://kino.pub/item/view/120646/s1e2" → "https://kino.pub/item/view/120646"
function stripEpisodeFromUrl(url) {
  if (!url) return '';
  return url.replace(/\/s\d+e\d+.*$/i, '');
}
